﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using Inet.Viewer.Data;
using Inet.Viewer.Resources;

namespace Inet.Viewer.WinForms.Prompt
{
    /// <summary>
    /// Panel for a time prompt field.
    /// </summary>
    public partial class TimePromptField : PromptControl
    {
        private static readonly CustomPromptValue customPromptValue = new CustomPromptValue();
        private DateTime oldValueIfNoValueNotChecked;

        /// <summary>
        /// Creates the panel.
        /// </summary>
        /// <param name="promptData">prompt field this panel is based one</param>
        /// <param name="validationDelegate">validation delegate for checking prompt values</param>
        /// <param name="hideNoValue">flag indicating the no value checkbox should be hidden</param>
        public TimePromptField(PromptData promptData, ValidationDelegate validationDelegate, bool hideNoValue)
        {
            this.PromptData = promptData;
            InitializeComponent();
            timePicker.ValueChanged += HandleChanged;
            PromptValue alreadySelectedValue = promptData.Values;
            if (alreadySelectedValue != null && alreadySelectedValue.Value is TimeSpan)
            {
                timePicker.Value = ToDateTime((TimeSpan)alreadySelectedValue.Value);
            }
            else if (!hideNoValue)
            {
                chkNoValue.Checked = true;
            }

            if (hideNoValue)
            {
                chkNoValue.Visible = false;
            }
            chkNoValue.Text = strings.Prompt_None_Long;
            ErrorProvider = new System.Windows.Forms.ErrorProvider(this.components);
            ControlForErrors = chkNoValue.Visible ? (Control)this.chkNoValue : (Control)this.timePicker;
        }

        /// <summary>
        /// Converts a TimeSpan to a DateTime.
        /// </summary>
        /// <param name="timeSpan">the time span</param>
        /// <returns>the date time</returns>
        public static DateTime ToDateTime(TimeSpan timeSpan)
        {
            return new DateTime(1900, 1, 1, timeSpan.Hours, timeSpan.Minutes, timeSpan.Seconds);
        }

        /// <summary>
        /// Converts the time of a DateTime to a TimeSpan.
        /// </summary>
        /// <param name="dt">the date time</param>
        /// <returns>the time span</returns>
        public static TimeSpan ToTimeSpan(DateTime dt)
        {
            return new TimeSpan(dt.Hour, dt.Minute, dt.Second);
        }

        /// <summary>
        /// Returns the chosen value for this panel. Never null: if null value, will return null wrapped in a SinglePromptValue.
        /// </summary>
        internal override PromptValue Value
        {
            get
            {
                return new SinglePromptValue(ToTimeSpan(this.timePicker.Value), null, PromptData.Type);
            }
            set
            {
                this.timePicker.Value = ToDateTime((TimeSpan)value.Value);
            }
        }

        /// <summary>
        /// Called when the "no value" checkbox is checked: sets the value to null
        /// </summary>
        /// <param name="sender">button being checked</param>
        /// <param name="e">args of the event</param>
        private void chkNoValue_CheckedChanged(object sender, EventArgs e)
        {
            if (this.chkNoValue.Checked)
            {
                oldValueIfNoValueNotChecked = this.timePicker.Value;
                this.timePicker.Enabled = false;
            }
            else
            {
                if (oldValueIfNoValueNotChecked == null)
                {
                    oldValueIfNoValueNotChecked = new DateTime();
                }
                this.timePicker.Value = oldValueIfNoValueNotChecked;
                this.timePicker.Enabled = true;
            }
        }
 
    }
}
